function [M_Garr,M_GParr,Marr,Garr,GParr] ...
    = reconstructCurrentsFFT_direct_MSD_woH(mu_0,d,px,data,imposeRefBC,p,lambdaArr,hArr,JxN,JyN)
% Use FFT to reconstruct currents directly from magnetic field and compare the performances 
% of the standard GCV, the projected GCV and the optimal solution
% minimizing the MSD. This code performs reconstructions on a large number
% of heights.
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%           * hArr - array of heights above the surface of the sample at
%           which to perform the reconstructions
%           * JxN,JyN - true x- and y- components of the current density.
%   Outputs: * M_Garr - array of MSD values obtained at different heights
%            by minimizing the GCV function
%            * M_GParr - array of MSD values obtained at different heights
%            by minimizing the projected SURE function
%            * Marr - array of the minimum MSD values obtaineable at
%            different heights.
%            * Garr - array of value of the GCV function at its minima at
%            different heights.
%            * GParr - array of value of the projected SURE function at its 
%            minima at different heights.

% tic
s2 = estimateNoiseK0(data); %estimate variance of noise perturbing the data
indexat = @(matrix,indices) matrix(indices{1},indices{2}); %auxilliary function to be used later for projections
vec = @(matrix) matrix(:); %vectorize a matrix

if imposeRefBC %impose boundary conditions
    offs=size(data);
    szO = size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
    szO = size(data);
end

if isscalar(px)
    px=[px,px];
end
sz = size(data);

% generate kernels as anonymous functions of the height:
K1 = @(h) genKers(h,1);
K2 = @(h) genKers(h,2);
KK = @(h) abs(K1(h)).^2+abs(K2(h)).^2;
KK_vec = @(h) vec(KK(h));

lf = calcLap(sz); %generate Laplacian convolution filter
lf = fft2(lf);
clf = abs(lf).^2;

%Get auxilliary quantity meant to avoid NaNs at (1,1) due to kernels. Analytically, this point should always be zero.
tmp = zeros(size(data)); 
tmp(1,1)=1;

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
jx = @(l,h) real(ifft2((conj(K1(h)))./(KK(h) + l*clf + tmp).*B));
jy = @(l,h) real(ifft2((conj(K2(h)))./(KK(h) + l*clf + tmp).*B));

auxRho = @(l,h) l*clf(:).*B(:)./(KK_vec(h) + l*clf(:) + tmp(:)); %analytic formula for squared residual norm
rho = @(l,h) norm(auxRho(l,h))^2/numel(B); %divide by extra factor of numel(B) from FFT2

data_rec = @(l,h) ifft2(K1(h).*fft2(jx(l,h))+K2(h).*fft2(jy(l,h))); %reconstructed data

secOffs = ceil(p*szO); %width of boundary layer to remove from P-SURE estimate
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)}; %indices for projections for P-SURE

rhoP = @(l,h) norm(indexat(data_rec(0,h)-data_rec(l,h),indsProj),'fro')^2;

auxTl = @(l,h) l*clf(:)./(KK_vec(h) + l*clf(:)+ tmp(:)); %analytic formula for trace of regularization operator.
T = @(l,h) sum(auxTl(l,h))/numel(B); %divide by extra factor of numel(B) from FFT2

noise = 2*randi(2,size(B))-3; %generate random matrix with entries +-1 with 50% probability 
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l,h) indexat(...
    ifft2(KK(h)./(KK(h) + l*clf + tmp).*N)...
,indsProj);
TP = @(l,h) sum(sum(abs(N_n).^2-conj(N_n).*B_n(l,h)));

G = @(x) rho(x(1),x(2))./((T(x(1),x(2))).^(2)); %define the GCV function w/ vector argument x = [l, h].
GP = @(x) rhoP(x(1),x(2)) - 2*s2*TP(x(1),x(2)); %define projected SURE function similarly to GCV.

if imposeRefBC %if data was reflected, discrad reflections in computing MSD
    indsProjMSD = {1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2)};
    MSD = @(l,h) (norm(indexat(jx(l,h),indsProjMSD)-JxN,'fro')^2+norm(indexat(jy(l,h),indsProjMSD)-JyN,'fro')^2)...
        /(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
else
    MSD = @(l,h) (norm(jx(l,h)-JxN,'fro')^2+norm(jy(l,h)-JyN,'fro')^2)/(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
end

%initialize arrays for MSD, GCV and P-SURE values:
M_Garr=zeros(length(hArr),1);
M_GParr = M_Garr; Marr = zeros(length(hArr),1);
Garr = zeros(length(hArr),1); GParr = Garr;

for jj=1:length(hArr) %for each height, find lambda with each parameter-choice method
    [Garr(jj),alphaG] = findGlobalMinV1(lambdaArr,@(l)G([l,hArr(jj)]),0); %Find global minimum of GCV and save only the corresponding lambda
    [GParr(jj),alphaGP] = findGlobalMinV1(lambdaArr,@(l)GP([l,hArr(jj)]),0);
    [Marr(jj),~] = findGlobalMinV1(lambdaArr,@(l)MSD(l,hArr(jj)),0);
    M_Garr(jj) = MSD(alphaG,hArr(jj));
    M_GParr(jj) = MSD(alphaGP,hArr(jj));
end

% toc
function K = genKers(h,n)
    switch n
        case 1
            [K,~] = TakeDFTOfK1K2(mu_0,h,d,sz,px);
        case 2
            [~,K] = TakeDFTOfK1K2(mu_0,h,d,sz,px);
        otherwise
            error('Invalid kernel number')
    end
end

end