function [M_Garr,M_GParr,Marr,Garr,GParr] ...
    = reconstructCurrentsFFT_stream_MSD_woH(mu_0,d,px,data,imposeRefBC,p,lambdaArr,hArr,JxN,JyN)
% Use FFT to reconstruct currents using the auxilliary function g(x,y) such that j=curl(g \hat{z}) and compare the performances 
% of the standard GCV, the projected GCV and the optimal solution
% minimizing the MSD. This code performs reconstructions on a large number
% of heights.
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%           * hArr - array of heights above the surface of the sample at
%           which to perform the reconstructions
%           * JxN,JyN - true x- and y- components of the current density.
%   Outputs: * M_Garr - array of MSD values obtained at different heights
%            by minimizing the GCV function
%            * M_GParr - array of MSD values obtained at different heights
%            by minimizing the projected SURE function
%            * Marr - array of the minimum MSD values obtaineable at
%            different heights.
%            * Garr - array of value of the GCV function at its minima at
%            different heights.
%            * GParr - array of value of the projected SURE function at its 
%            minima at different heights.

% tic
s2 = estimateNoiseK0(data); %estimate variance of noise perturbing the data
indexat = @(matrix,indices) matrix(indices{1},indices{2}); %auxilliary function to be used later for projections
vec = @(matrix) matrix(:); %vectorize a matrix

if imposeRefBC %impose boundary conditions
    offs=size(data);
    szO = size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
end

if isscalar(px)
    px=[px,px];
end
sz = size(data);

% generate kernels
K = @(h) TakeDFTOfKF(mu_0,h,d,sz,px);
KK = @(h)abs(K(h)).^2;
KK_vec = @(h)vec(KK(h));

lf = calcLap(sz);
lf = fft2(lf);
clf = abs(lf).^2;

%Get auxilliary quantity meant to avoid NaNs at (1,1) due to kernels. Analytically, this point should always be zero.
tmp = zeros(size(data));
tmp(1,1)=1;

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
g = @(l,h) real(ifft2((conj(K(h)))./(KK(h) + l*clf + tmp).*B));

auxRho = @(l,h) l*clf(:).*B(:)./(KK_vec(h) + l*clf(:) + tmp(:));
rho = @(l,h) norm(auxRho(l,h))^2/numel(B);

data_rec = @(l,h) ifft2(K(h).*fft2(g(l,h))); %reconstructed data

secOffs = ceil(p*szO); %width of boundary layer to remove from P-SURE estimate
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)}; %indices for projections for P-SURE
rhoP = @(l,h) norm(indexat(data_rec(0,h)-data_rec(l,h),indsProj),'fro')^2;

noise = 2*randi(2,size(B))-3; %generate random matrix with entries +-1 with 50% probability 
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l,h) indexat(...
    ifft2(KK(h)./(KK(h) + l*clf + tmp).*N)...
,indsProj);
TP = @(l,h) sum(sum(abs(N_n).^2-conj(N_n).*B_n(l,h)));

auxTl = @(l,h) l*clf(:)./(KK_vec(h) + l*clf(:)+ tmp(:));
T = @(l,h) sum(auxTl(l,h))/numel(B);
G = @(x) rho(x(1),x(2))./((T(x(1),x(2))).^2); %define the GCV function w/ vector argument x = [l, h].

GP = @(x) rhoP(x(1),x(2)) - 2*s2*TP(x(1),x(2)); %P-SURE function w/ vector argument x = [l, h].

[L1,L2] = generateSGolayDiffFilter(size(data),2,5,0); %Savitsky-Golay differentiation filter with polynomial order 2 and window length 5
jx = @(l,h) L1*g(l,h)/px(1); %differentiate g(x,y) to get currents 
jy = @(l,h) -g(l,h)*L2'/px(2);

if imposeRefBC %if data was reflected, discrad reflections in computing MSD
    indsProjMSD = {1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2)};
    MSD = @(l,h) (norm(indexat(jx(l,h),indsProjMSD)-JxN,'fro')^2+norm(indexat(jy(l,h),indsProjMSD)-JyN,'fro')^2)...
        /(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
else
    MSD = @(l,h) (norm(jx(l,h)-JxN,'fro')^2+norm(jy(l,h)-JyN,'fro')^2)/(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
end

%initialize arrays for MSD, GCV and P-SURE values:
M_Garr=zeros(length(hArr),1);
M_GParr=zeros(length(hArr),1);
Garr = zeros(length(hArr),1);
GParr = zeros(length(hArr),1);
Marr = zeros(length(hArr),1);

for jj=1:length(hArr) %for each height, find lambda with each parameter-choice method
    [Garr(jj),alphaG] = findGlobalMinV1(lambdaArr,@(l)G([l,hArr(jj)]),0); %Find global minimum of GCV and save only the corresponding lambda
    [GParr(jj),alphaGPP] = findGlobalMinV1(lambdaArr,@(l)GP([l,hArr(jj)]),0); %Find global minimum of GCV and save only the corresponding lambda
    [Marr(jj),~] = findGlobalMinV1(lambdaArr,@(l)MSD(l,hArr(jj)),0);
    M_Garr(jj) = MSD(alphaG,hArr(jj));
    M_GParr(jj) = MSD(alphaGPP,hArr(jj));
end

%toc