function [X_MAP,S_MAP,k_MAP] = BM_unitary_MAP_Gibbs_sampler(Y,var_e,model_params,num_rounds,beta0)
%BM_unitary_MAP_Gibbs_sampler aproximates the BM-based MAP estimator of the 
%representation coefficients for a unitary dictionary via Gibbs sampling and 
%simulated annealing. 
% BM - Boltzmann Machine
% BM-based model - BM prior for the sparsity pattern and Gaussian nonzero coefficients.
% =====================================================================================
% Input:
% Y - an n-by-N matrix consisting of N noisy signals
% var_e - the variance of the gaussian distribution of the additive noise
% model_params - parameters for the stochastic model. Required fields:
% model_params.dictionary - a matrix of size n-by-n consisting of the dictionary atoms.
% The dictionary should be square and unitary.
% model_params.variances - a vector of size n-by-1 consisting of the variances of the 
% gaussian distributions of the nonzero representation coefficients.
% model_params.W, model_params.b - the Boltzmann parameters: an interaction matrix 
% of size n-by-n and a bias vector of size n-by-1.
% num_rounds - number of rounds for the Gibbs sampler
% beta0 - parameter for the annealing schedule
% =====================================================================================
% Output:
% X_MAP - an n-by-N matrix consisting of the restored representation vectors
% S_MAP - an n-by-N matrix consisting of the sparsity patterns
% k_MAP - a vector of size N-by-1 consisting of the lengths of the recovered supports
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
if nargin<5
    beta0=1; % no annealing
end
[n,N]=size(Y);
A=model_params.dictionary;
% Check input parameters
G=A'*A;
X_MAP=[];
S_MAP=[];
k_MAP=[];
if size(A,2)~=n
    display('Invalid input - the dictionary should be square')
    return;
elseif sum(sum(abs(G-eye(n))>1e-10))>0
    display('Invalid input - the dictionary should be unitary')
    return;
end
W=model_params.W;
b=model_params.b;
var_x=model_params.variances;
% Gibbs sampling from a BM distribution + simulated annealing
T0=beta0^(n*num_rounds-1); % initial temperature
S_MAP=-1*ones(n,N);
hh = waitbar(0,'Approximating MAP via Gibbs sampling and simulated annealing');
for l=1:N
    if ~rem(l,100)
        waitbar(l/N,hh)
    end
    y=Y(:,l);
    q=b+0.25*(var_x./(var_e*(var_e+var_x)).*(A'*y).^2-log(1+var_x/var_e));
    S=2*(rand(n,1)>0.5)-1; % initialize randomly using uniform distribution
    T=T0; % initialize temperature
    for k=1:num_rounds
        seq_inds=randperm(n);
        for i=1:n
            curr_ind=seq_inds(i);
            Pr=1./(1+exp(2*S(curr_ind)*(W(curr_ind,:)*S+q(curr_ind)))/T);
            cond=rand(1)<Pr;
            S(curr_ind)=S(curr_ind)*(1-2*cond);
            T=T/beta0; % update temperature - annealing schedule
        end
    end
    S_MAP(:,l)=S;
end
close(hh)
k_MAP=sum(S_MAP==1);
% Compute estimates for the represenations using the oracle formula
X_MAP=unitary_oracle_formula(Y,var_e,model_params,S_MAP);