function [X_OMP,S_OMP,k_OMP] = OMP(Y,A,stop_method,stop_params)
%OMP finds an approximate solution to the problem: 
%argmin_x ||x||_0 s.t. ||y-Ax||_2<=epsilon
% OMP - Orthogonal Matching Pursuit
% This is a fast version of the algorithm which makes efficient computations
% of inverse matrices using algebric shortcuts. 
% =====================================================================================
% Input:
% Y is an n-by-N matrix consisting of N signals. 
% A is a matrix of size n-by-m consisting of the dictionary atoms (normalized).
% stop_method,stop_params - variables for the stopping rule.
% =====================================================================================
% Output:
% X_OMP is an m-by-N matrix consisting the restored representation vectors.
% S_OMP is an m-by-N matrix consisting the sparsity patterns.
% k_OMP is a vector of length N consisting the lengths of the recovered supports.
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
N=size(Y,2);
[n,m]=size(A);
X_OMP=zeros(m,N);
S_OMP=-1*ones(m,N);
k_OMP=zeros(1,N);
hh = waitbar(0,'Performing OMP');
for l=1:N
    if ~rem(l,100)
        waitbar(l/N,hh)
    end
    s_recov=[];
    x_recov=zeros(m,1);
    y=Y(:,l);
    r=y;
    norm_r=norm(r);
    if strcmp(stop_method,'k_reached')
        err=1e-3;
        maxNumCoef=stop_params.k_true(l); 
    elseif strcmp(stop_method,'err_thr')
        err=stop_params.err_noise;
        maxNumCoef=n/2;
    end
    i=0;
    %Main Iteration
    while norm_r>err && i < maxNumCoef
        [~,add2s]=max(abs(A'*r));
        add2s=add2s(1);
        if i>0
            inv_A=inv_prev;
            B=A(:,s_recov)'*A(:,add2s);
            F=A(:,add2s)'*A(:,add2s);
        end
        s_recov=[s_recov,add2s];  
        if i==0
            inv_curr=1/(A(:,s_recov)'*A(:,s_recov));
        else
            V1=1/(F-B'*inv_A*B);
            V2=inv_A+inv_A*B*V1*B'*inv_A;
            V3=-inv_A*B*V1;
            inv_curr=[V2,V3;V3',V1];
        end
        x_s=inv_curr*A(:,s_recov)'*y;
        x_recov(s_recov)=x_s;
        r=y-A(:,s_recov)*x_s;
        norm_r=norm(r);
        i=i+1;
        inv_prev=inv_curr;
    end
    X_OMP(:,l)=x_recov(:);
    S_OMP(s_recov,l)=1;
    k_OMP(l)=numel(s_recov);
end
close(hh) 